<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Vendor;
use App\Models\Booking;
use App\Models\CarBooking;

class RecalculateVendorBalances extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vendors:recalculate-balances {--vendor_id=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate vendor balances based on paid bookings';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Recalculating vendor balances...');
        
        $vendorId = $this->option('vendor_id');
        
        if ($vendorId) {
            $vendors = Vendor::where('id', $vendorId)->get();
            if ($vendors->isEmpty()) {
                $this->error("Vendor with ID {$vendorId} not found");
                return Command::FAILURE;
            }
        } else {
            $vendors = Vendor::all();
        }
        
        $this->info("Processing {$vendors->count()} vendor(s)...");
        $bar = $this->output->createProgressBar($vendors->count());
        $bar->start();
        
        foreach ($vendors as $vendor) {
            // Calculate total from property bookings
            $propertyEarnings = Booking::where('vendor_id', $vendor->id)
                ->where('payment_status', 'paid')
                ->sum('vendor_payout_amount');
            
            // Calculate total from car bookings (rental payout)
            $carEarnings = CarBooking::where('vendor_id', $vendor->id)
                ->where('payment_status', 'paid')
                ->sum('vendor_payout_amount');
            
            // Calculate security deposit deductions approved by admin
            $depositDeductions = CarBooking::where('vendor_id', $vendor->id)
                ->whereIn('deposit_status', ['deducted', 'partially_deducted'])
                ->sum('deposit_deduction_amount');
            
            $totalEarnings = $propertyEarnings + $carEarnings + $depositDeductions;
            
            // Calculate total payouts already made
            $totalPayouts = $vendor->payouts()
                ->whereIn('status', ['approved', 'completed'])
                ->sum('amount');
            
            $correctPendingBalance = $totalEarnings - $totalPayouts;
            
            $this->newLine();
            $this->info("Vendor #{$vendor->id} ({$vendor->business_name}):");
            $this->info("  Property Earnings: ₦" . number_format($propertyEarnings));
            $this->info("  Car Rental Earnings: ₦" . number_format($carEarnings));
            $this->info("  Security Deposit Deductions: ₦" . number_format($depositDeductions));
            $this->info("  Total Earnings: ₦" . number_format($totalEarnings));
            $this->info("  Total Payouts: ₦" . number_format($totalPayouts));
            $this->info("  Current Pending Balance: ₦" . number_format($vendor->pending_balance));
            $this->info("  Correct Pending Balance: ₦" . number_format($correctPendingBalance));
            
            if ($vendor->pending_balance != $correctPendingBalance) {
                $this->warn("  Updating balance...");
                $vendor->update([
                    'pending_balance' => $correctPendingBalance,
                    'total_earned' => $totalEarnings,
                ]);
                $this->info("  ✓ Balance updated!");
            } else {
                $this->info("  ✓ Balance is correct");
            }
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        $this->info('Vendor balances recalculated successfully!');
        
        return Command::SUCCESS;
    }
}
