<?php

namespace App\Console\Commands;

use App\Models\CarBooking;
use Illuminate\Console\Command;

class RecalculateCarBookingCommissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'car-bookings:recalculate-commissions';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate commission and vendor payout amounts for existing car bookings';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Recalculating commission and vendor payout amounts for car bookings...');

        // Get all car bookings that don't have commission calculated
        $bookings = CarBooking::whereNull('commission_rate')
            ->orWhereNull('vendor_payout_amount')
            ->with('vendor')
            ->get();

        if ($bookings->isEmpty()) {
            $this->info('No bookings found that need recalculation.');
            return 0;
        }

        $this->info("Found {$bookings->count()} booking(s) to update.");

        $bar = $this->output->createProgressBar($bookings->count());
        $bar->start();

        $updated = 0;

        foreach ($bookings as $booking) {
            // Calculate commission and vendor payout
            $commissionRate = $booking->vendor->commission_rate ?? 10; // Default 10%
            $commissionAmount = ($booking->rental_cost * $commissionRate) / 100;
            $vendorPayoutAmount = $booking->rental_cost - $commissionAmount;

            // Update booking
            $booking->update([
                'commission_rate' => $commissionRate,
                'commission_amount' => $commissionAmount,
                'vendor_payout_amount' => $vendorPayoutAmount,
            ]);

            $updated++;
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);
        $this->info("Successfully updated {$updated} booking(s)!");

        return 0;
    }
}
